# Automatisierte Tests

Um die korrekte Implementierung der verschiedenen Praktika einfacher
überprüfen zu können, wird das [Google Test](https://google.github.io/googletest/)
Framework verwendet. Dieses ist platformübergreifend anwendbar, d.h. es
compiliert sowohl unter Linux, Mac, als auch Windows.

## Begrifflichkeiten

* Test Suit: Eine Ansammlung an Tests, die miteinander in Verbindung stehen.
* Test: Ein einzelner Testfall. Andere Frameworks verwenden hier auch den Begriff Test Case.
* Assertion: Ein Ausdruck der überprüft, ob eine Bedingung erfüllt ist oder nicht.

## Konzept

Um eine bestimmte Funktionalität zu testen, werden ein oder mehrere Assertions in einem
Test ausgeführt. Jeder Test sollte dabei möglichst auf einen Fall beschränkt sein.
Mehrere zusammengehörende Tests werden in einer Test Suit zusammengefasst. Eine Test
Suit sollte im Idealfall die Struktur des zu testenden Codes widerspiegeln.

## Struktur

Die CMakeLists.txt Datei innerhalb des src Ordners definiert die Option AK\_COMPILE\_TEST,
mit welcher gesteuert werden kann, ob die Tests integriert werden sollen oder nicht, und
enthält weiterhin folgenden Code.

```
if(AK_COMPILE_TESTS)
    # Deklare a dependency on GoogleTest which is downloaded from GitHub automatically.
    include(FetchContent)
    FetchContent_Declare(
      googletest
      # This commit hash should be updated regularly.
      URL https://github.com/google/googletest/archive/bf66935e07825318ae519675d73d0f3e313b3ec6.zip
    )
    # For Windows: Prevent overriding the parent project's compiler/linker settings
    set(gtest_force_shared_crt ON CACHE BOOL "" FORCE)
    FetchContent_MakeAvailable(googletest) 
    
    # All tests are located within the `src/test` directory.
    add_subdirectory(${AK_TEST_PATH})
    enable_testing()
endif(AK_COMPILE_TESTS)
```

Sollte `AK_COMPILE_TESTS` auf ON gesetzt sein, so wird cmake, falls noch nicht geschehen,
automatisch die in der URL angegebenen Version von Github herunterladen und in des
Projekt integrieren. Weiterhin wird das `AK_TEST_PATH` Verzeichnis, welches auf `src/test`
zeigen sollte, als Unterverzeichnis hinzugefügt.

Innterhalb von `src/test` befindet sich der Test-Runner, sowie die `CMakeLists.txt` zum
erstellen der Test-Executable. Die eigentlichen Tests befinden sich jeweils im zugehörigen
Praktikumsordner, innerhalb eines Unterverzeichnisses mit dem Namen `tests`. Um die Dateien
vor Manipulation zu schützen wird vor dem Verteilen der Praktika eine Checksumme berechnet,
die vor dem Testen der Praktika überprüft wird.

### Bibliotheken

Um die Funktionalität der einzelnen Funktionen und Klassen eines Praktikums zu testen,
muss auf diese zugegriffen werden können. Hierfür muss jedes Praktikum die entsprechen-
den Dateien in einer eigenen Bibliothek zusammenführen.

```
if (AK_COMPILE_TESTS)
    set (AK_Praktikum_CryptoPP_Present ON PARENT_SCOPE)
    ADD_LIBRARY(cryptopp−intro−test PRNG.cpp BlumBlumShubGenerator.cpp)
    TARGET_LINK_LIBRARIES(cryptopp−intro−test ${CRYPTOPP_LIBRARY} ${CMAKE_THREAD_LIBS_INIT})
endif (AK_COMPILE_TESTS)
```

So werden z.B. die beiden Dateien `PRNG.cpp` und `BlumBlumShubGenerator.cpp`, des
Praktikum-CryptoPP Praktikums, in einer eigenen Library zusammengefasst.
Man bemerke dabei das Setzen der Variable `AK_Praktikum_CryptoPP_Present`. Hiermit
wird später bestimmt, welche Testdateien benötigt werden, je nachdem ob das Praktikum
existiert oder nicht.

### Einbinden der Bibliotheken

Die `CMakeLists.txt` Datei von `src/test` erstellt einen Test-Runner, d.h. eine ausführbare
Datei, die die einzelnen Tests durchführt, und führt diesen, nach jedem Kompilieren,
automatisch aus.

```
# src/test/CMakeLists.txt

# List of all test files (one for each exercise).
set(AK_TEST_EXECUTABLES)
# List of all exercise libraries which expose the respective interface.
set(AK_TEST_LIBRARIES)
# List of the exercise headers.
set(AK_TEST_HEADERS)

if(AK_Praktikum_CryptoPP_Present)
    #list(APPEND AK_TEST_EXECUTABLES cryptopp-test.cpp)
    list(APPEND AK_TEST_EXECUTABLES ../Praktikum-CryptoPP/tests/cryptopp-test.cpp)
    list(APPEND AK_TEST_LIBRARIES cryptopp-intro-test)
    list(APPEND AK_TEST_HEADERS ../Praktikum-CryptoPP/)

    message(STATUS "Praktikum-CryptoPP tests added")
endif(AK_Praktikum_CryptoPP_Present)
```

Auf folgende Besonderheiten muss bei dem Hinzufügen neuer Testdateien geachtet werde:

1. Jeder zu testende Praktikumsordner sollte zu `AK_TEST_HEADERS` hinzugefügt
werden, um die darin enthaltenen .h Dateien direkt einbinden zu können.
2. Jede Testdatei muss zu `AK_TEST_EXECUTABLES` hinzugefügt werden. 
3. Die Bibliotheken der zu testenden Praktika müssen unter `AK_TEST_LIBRARIES`
hinzugefügt werden

Das hinzufügen der einzelnen Bibliotheken und Pfade für die Praktika, sollte bedingt
erfolgen. Hierfür wird die `AK_Name_des_Praktikums_Present` Variable geprüft. Sollte ein
Praktikum (noch) nicht existieren, so wird die zugehörige Testdatei entsprechend nicht
mit eingebunden. Außerdem werden so Fehler vermieden, die auf die Nichtexistenz von
(Praktikums-)Bibliotheken zurückzuführen ist.

Mithilfe von `gtest_discover_tests(ak_test)` werden alle definierten Tests zur Laufzeit
durch eine automatisch von GoogleTest generierte main Funktion aufgerufen.

Die letzte Zeile von `src/test/CMakeLists.txt` sorgt dafür, dass nach einem erfolgreichen 
Kompilieren alle Tests automatisch ausgeführt werden, was etwas Zeit bei der Implementierung
der Praktika spart.

```
# Run tests after each successful make build.
option(AK_AUTORUN_TESTS "Run tests after successful build" ON)
if(AK_AUTORUN_TESTS)
add_custom_command(TARGET ak_test COMMAND ../bin/ak_test POST_BUILD)
endif(AK_AUTORUN_TESTS)
```

> Für docker muss diese Funktion deaktiviert werden, da bei fehlschlagenden Tests sonst
> abgebrochen wird!

## Tests

Um eine Funktion, Methode oder Klasse zu testen, müssen Annahmen über deren Verhal-
ten getroffen werden. Diese werden durch sog. Assertions ausgedrückt. Bei diesen handelt
es sich um Makros die z.B. dafür verwendet werden können, einen berechneten Wert mit
einem Erwartungswert zu vergleichen.

Man kann dabei zwischen zwei Arten von Assertions unterscheiden, nämlich denen die bei
einem Fehler vorzeitig abbrechen (diese beginnen mit `ASSERT_*`) und denen die bei einem
Fehler zur nächsten Assertion übergehen (diese beginnen mit `EXPECT_*`). `ASSERT_*` sollte
immer dann verwendet werden, wenn es für einen Test keinen Sinn macht fortzufahren,
z.B. weil durch den Fehler davon ausgegangen werden kann, dass sich eine Klasse in
einem undefinierten Zustand befindet und die nachfolgenden Assertions auf der vorherigen
aufbauen. Normalerweise ist `EXPECT_*` die bessere Wahl.

### Unit Test

Um eine einfache Testfunktion zu definieren, wird das `TEST()` Makro verwendet. Dieses
kann wie eine normale Funktion betrachtet werden, die void zurückliefert. Ob ein Test erfolgreich 
ist hängt von den in dem Test verwendeten Assertions ab. Sollten alle Assertions
erfüllt sein, so ist der Test erfolgreich, ansonsten nicht.

```
TEST(MillerRabinTestTest, Witness) {
    Integer n { "121399589920585397786647347761633" };
    Integer witness { "3" };
    EXPECT_EQ(MillerRabinTest::witness(witness, n), true);
}
```

Der obige Code ist aus `primzahlen-test.cpp` entnommen und definiert einen neuen Test
mit dem Namen `Witness` für die Suit `MillerRabinTestTest`, unter welcher alle Tests der
MillerRabinTest-Klasse zusammengefasst werden.

Mit diesem wird getestet, ob die `witness()` Methode korrekt erkennt, dass die Zahl
`121399589920585397786647347761633` zusammengesetzt ist. Hierfür wir die Assertion
`EXPECT_EQ` verwendet und das Ergebnis des Methodenaufrufs mit true verglichen, da
in diesem Fall erwartet wird, dass die Methode für den Zeugen `3` erkennt, dass `n` eine
zusammengesetzte Zahl ist.
