/*
 * SPN.h
 *
 *      Author: Prof. Dr. Christoph Karg (Hochschule Aalen)
 */

#ifndef SPN_H_
#define SPN_H_

#include <vector>
#include <vector>
#include <SPN/SBoxArray.h>
#include <SPN/Permutation.h>

using namespace std;

class SPN {
protected:
	/// Id of the SPN
	string id;
	/// A short description of the SPN (optional)
	string description;
	/// Bit length of the SPN
	ushort bit_length;
	/// Number of rounds of the SPN
	ushort nr_of_rounds;
	/// Array containing the subkeys (size must be equal to nr_of_rounds+1)
	vector<vector<bool> > subkey_array;
	/// Array containing the sbox arrays (size must be equal to nr_of_rounds)
	vector<SBoxArray> sbox_array;
	/// Array containing the permutations (size must be equal to nr_of_rounds-1)
	vector<Permutation> perm_array;
	/// Array containing the inverse permutations (size must be equal to nr_of_rounds-1)
	vector<Permutation> inv_perm_array;

public:
	SPN();
	SPN(const SPN& spn);
	virtual ~SPN();

	virtual void set(const string& id, const vector<SBoxArray>& sbox_array, const vector<Permutation>& perm_array);

	void setKey(const vector<bool>& key);

	virtual vector<bool> encrypt(const vector<bool>& block) const;

	virtual vector<bool> decrypt(const vector<bool>& block) const;

	virtual vector<byte> encrypt(const vector<byte>& plain_text) const;

	virtual vector<byte> decrypt(const vector<byte>& cipher_text) const;

	/// Get the number of required key bits.
	ushort getKeyLength() const;
	/// Get the bit length of the SPN
	ushort getBitLength() const;
	/// Get the bit length of the SPN's sbox
	ushort getSBoxBitLength() const;
	/// Get the description of the SPN
	string getDescription() const;
	/// Get the number of rounds of the SPN
	ushort getNrOfRounds() const;
	// Set the description of the SPN
	void setDescription(string d);
	/// Assignment operator.
	SPN& operator=(const SPN& spn);
	/// Get the final s-box layer for analysis purposes
	SBoxArray getFinalSBoxArray() const;

};

#endif /* SPN_H_ */
